import processing.core.*; 
import processing.data.*; 
import processing.event.*; 
import processing.opengl.*; 

import processing.serial.*; 

import java.util.HashMap; 
import java.util.ArrayList; 
import java.io.File; 
import java.io.BufferedReader; 
import java.io.PrintWriter; 
import java.io.InputStream; 
import java.io.OutputStream; 
import java.io.IOException; 

public class Proc_Servo_Ctrl_05 extends PApplet {

// ######################################################################
//
//  Servo Control File v0.05 Beta
//
//  Released:  13/02/2017    By: Tony Swanwick
//
// ######################################################################
/*
  This code communicates with the Arduino UNO to controls a servo via
  the USB serial interface.
  
  This version includes:
  You can now adjust slide upper and lower limits.
  Tick link active every 500ms
  
*/

// declare libraries

Serial usbPort;

// declare variables

int angleMode; // 0 = degrees, 1 = microseconds
int comFlag = 0; // =1 if COM serial available
int comListLength = 0;  // number of items in COM connection list
String comName = ""; // serial port COM name
int comPnt = 0; // COM port list pointer
int ctrlMode; // control mode, 0 = slider, 1 = square, 2 = triangle, 3 =sine
int cycles; // waveform counter
String data = "";
int drawFlag = 1; // = 1 to force screen redraw
float zF; // any temp float value
PImage img; // image loaded as background
int index = 0;
int interval = 10; // main loop interval in milliseconds
int JoyX0 = 170; // x-coordinate of joystick slider left end
int JoyX1 = 630; // x-coordinate of joystick slider right end
int JoyXval; // joystick slider x-value
int keyDwn = 0; // tracks keyboard up/down activity, 0 = up, 1 = down
int LF = 10; // ASCII line feed
int mDown,mON,mTO,mX, mY;  // mouse flag and last x,y
String msgCh; // channel message sent to Arduino
String msgCOM = "None"; // COM port name
String msgRx = ""; // message received from Arduino
String msgTx = ""; // message sent to Arduino
int nextMillis = 0; // loop end time in millis
int period; // half period in draw loop units
int phase; // current waveform phase
int pingON = 1; // > 1 sends regular '~' pings
int qPhase; // quiver phase, 0 or 1
int quiverFlag; // Quiver mode 0 = OFF, 1 = ON
int qVal; // applied quiver offset
int refreshFlag = 1; // = 1 at end of sweep to redraw background
int servoMin, servoMax; // limit constants for servo values
int servoMMD; // limit difference constant for servo values
int servoPin; // pin number used by Arduino
int servoLast = 0; // previous value sent to servo
int servoSwp; // servo sweep flag; default = 0  OFF, 1 = ON
int servoVal; // servo value for manual setting
int servoVLL; // servo value LL for manual setting
int servoVUL; // servo value UL for manual setting
int sliderDwn; // =1 when mouse is clicked on slider
int TB = 0;  // text Blue colour component
int TG = 0;  // text Green colour component
int tick = 0; // tick link send counter
int timeCnt; // wave count down timer
int TMC; // 50ms timer
int TR = 0;  // text Red colour component
int upDwnFlg = 0; // up/down state flag, 0 = down, 1 = up

public void setup() {
  // first code to run
  mON = 0; // mouse X,Y is OFF initially
  resetFlags();
    // screen size must match background bitmap
  loadScrn(0); // load the default screen
  getCOMPort();
  // allow Arduino time to boot up
  delay(100);
  centreJoyX();
  println("Ready!");
}

public void draw() {
  // main loop. Comes here about 100 times a second
  TMC--; if (TMC < 1) {
    // do this every 3 loop counts, 33 Hz
    TMC = 2;
    // perform mode actions
    switch(ctrlMode) {
      case 0:
        // slider mode, check for quiver
        if (quiverFlag > 0) {doQuiver();}
        else if (quiverFlag < 0) {setQuiverOFF();}
        break;
      case 1: doSquare(); break;
      case 2: doTriangle(); break;
      case 3: doSine(); break;
      case 4: doSwing(); break;
    }
  }
  if (mDown > 0) {mTO--; if (mTO < 1) {mouseRepeat(); mTO = 4;}}
  if (mON > 0) {drawMouseXY();}
  if (drawFlag > 0) {
    background(img); // define background RGB colour
    drawChMsg(); drawServoVal(); drawLamps();
    drawLLMsg(); drawULMsg(); drawJoyX();
    drawCOMMsg(); drawRxMsg(); drawTxMsg();
    drawFreq(); drawCycles();
    drawFlag = 0;
  }
  // now wait for end of 10ms loop time
  while (millis() < nextMillis) {
    // wait for remainder of 10ms period to sync overall period
  } nextMillis = millis() + interval;
  if (pingON > 0) {
    tick--; if (tick < 1) {
      tick = 50; // reset tick period to 500ms
      usbPortWrite("~"); // send a tick
    }
  }
}

// ----------------------------------------------------------------------

public void keyPressed() {
  // used to receive keyboard keys
  keyDwn = 1;
}

// ----------------------------------------------------------------------

public void keyReleased() {
  keyDwn = 0;
}

// ----------------------------------------------------------------------

public void mouseDragged() {
  // mouse has been clicked and dragged on window
  mX = mouseX; mY = mouseY;
  if (sliderDwn > 0) {
    // ignore mY if it goes outside of slider region
    if ((mX >=150) && (mX <=650)) {
      // mouse in X-slider region
      readXSlider();
    }
  }
}

// ----------------------------------------------------------------------

public void mouseMoved() {
  // called whenever the mouse moves
  mX = mouseX; mY = mouseY;
}

// ----------------------------------------------------------------------

public void mousePressed() {
  // mouse has been clicked on window
//  mX = mouseX; mY = mouseY;
  mDown = 1; mTO = 30;
  if ((mX < 26) && (mY < 26)) {toggleMouseON();}
  // pin number region?
  if ((mX > 198) && (mX < 552) && (mY > 63) && (mY < 144)) {sendPin();}
  // angle mode buttons
  if ((mX > 575) && (mX < 664)) {
    if ((mY > 63) && (mY < 104)) {setAngleMode(0);}
    if ((mY > 103) && (mY < 144)) {setAngleMode(1);}
  }
  // ping button
  if ((mX > 672) && (mX <760) && (mY > 103) && (mY < 144)) {togglePing();}
  // slider region?
  if ((mX >=136) && (mX <155) && (mY >= 162) && (mY <= 196)) {decXSlider();}
  if ((mX >645) && (mX <=663) && (mY >= 162) && (mY <= 196)) {incXSlider();}
  if ((mX >=155) && (mX <=645) && (mY >= 162) && (mY <= 196)) {
    // mouse in X-slider region
    sliderDwn = 1; readXSlider();
  }
  if ((mY > 163) && (mY < 195)) {
    // adjust slider upper and lower limits
    if ((mX > 34) && (mX < 52)) {setLLValDwn();}
    if ((mX > 112) && (mX < 129)) {setLLValUp();}
    if ((mX > 669) && (mX < 686)) {setULValDwn();}
    if ((mX > 749) && (mX < 764)) {setULValUp();}
  }
  if ((mY > 167) && (mY <= 190)) {
    // store upper and lower limits
    if ((mX >= 53) && (mX <= 112)) {setLL();}
    if ((mX >= 686) && (mX <= 746)) {setUL();}
  }
  if ((mX > 34) && (mX < 115) && (mY > 62) && (mY < 144)) {sendRESET();}
  if ((mX > 384) && (mX < 415) && (mY > 198) && (mY < 222)) {centreServo();}
  if ((mX > 88) && (mX < 176) && (mY > 231) && (mY < 256)) {getCOMPort();}
  // mode buttons
  if ((mY > 277) && (mY < 375)) {
    if ((mX > 34) && (mX < 133)) {setMode(0);}
    if ((mX > 135) && (mX < 233)) {setQuiver();}
    if ((mX > 235) && (mX < 334)) {setMode(1);}
    if ((mX > 336) && (mX < 435)) {setMode(2);}
    if ((mX > 437) && (mX < 536)) {setMode(3);}
    if ((mX > 537) && (mX < 636)) {setMode(4);}
  }
  // check period buttons
  if ((mY > 295) && (mY < 328)) {
    if ((mX > 669) && (mX < 686)) {decPeriod();}
    if ((mX > 685) && (mX < 748)) {period = 20; drawFlag = 1;}
    if ((mX > 747) && (mX < 764)) {incPeriod();}
  }
  if ((mX > 669) && (mX < 764) && (mY > 333) && (mY < 359)) {cycles = 0; drawFlag = 0;}
}

// ----------------------------------------------------------------------

public void mouseReleased() {
  // mouse released from click and drag operation
  mX = mouseX; mY = mouseY;
  mDown = 0;
  sliderDwn = 0; // clear slider flag
}

// ----------------------------------------------------------------------

public void mouseRepeat() {
  // called if mouse still down after a pause period
  mX = mouseX; mY = mouseY;
  if (sliderDwn < 1) {
    if ((mX >=136) && (mX <155) && (mY >= 162) && (mY <= 196)) {decXSlider();}
    if ((mX >645) && (mX <=663) && (mY >= 162) && (mY <= 196)) {incXSlider();}
    if ((mY > 163) && (mY < 195)) {
      // adjust slider upper and lower limits
      if ((mX > 34) && (mX < 52)) {setLLValDwn();}
      if ((mX > 112) && (mX < 129)) {setLLValUp();}
      if ((mX > 669) && (mX < 686)) {setULValDwn();}
      if ((mX > 749) && (mX < 764)) {setULValUp();}
    }
  }
  // check period buttons
  if ((mY > 295) && (mY < 328)) {
    if ((mX > 669) && (mX < 686)) {decPeriod();}
    if ((mX > 747) && (mX < 764)) {incPeriod();}
  }
}

// ----------------------------------------------------------------------

public void serialEvent(Serial usbPort) {
  // respond to a serial data received event
  data = usbPort.readStringUntil(LF);
  msgRx = trim(data); data = "";
  println(msgRx);
  drawFlag = 1;
}

// ----------------------------------------------------------------------

public void calcQV() {
  // calculate quiver offset value, approx +/- 5
  qVal = (servoVUL-servoVLL)/45;
}

// ----------------------------------------------------------------------

public void centreJoyX() {
  // centre the joystick based on the current limits, LL, UL
  servoVal = servoVLL + ((servoVUL - servoVLL)/2);
  setXSlider(); drawFlag = 1;
}

// ----------------------------------------------------------------------

public void centreServo() {
  // centre the servo based on the current limits, LL, UL
  centreJoyX(); servoLast = servoVal;
  sendAngle();
}

// ----------------------------------------------------------------------

public void decPeriod() {
  // decrement the period value
  if (period > 10) {period--; drawFlag = 1;}
}

// ----------------------------------------------------------------------

public void decXSlider() {
  // decrement slider value
  if (servoVal > servoVLL) {
    servoLast = servoVal;
    if (angleMode > 0) {
      servoVal--;
    } else {
      zF = servoVal;
      zF = 0.27f + (((zF - 1000.0f) * 180.0f) / 1000.0f); zF--;
      servoVal = PApplet.parseInt((zF * 1000.0f)/180.0f) + 1000; 
      servoVal = max(servoVal,servoVLL);
    } setXSlider(); sendAngle();
  }
}

// ----------------------------------------------------------------------

public void doQuiver() {
  // send quiver angles
  quiverFlag--;
  if (quiverFlag < 1) {
    quiverFlag = 5; // main timer base on draw loop timings
    if (qPhase < 1) {
      qPhase = 1;
      msgTx = "SM" + str(servoVal + qVal) + ".";
    } else {
      qPhase = 0;
      msgTx = "SM" + str(servoVal - qVal) + ".";
    }
    usbPortWrite(msgTx); drawFlag = 1;
  }
}

// ----------------------------------------------------------------------

public void doSine() {
  // send sinewave angles
  float zHA = (servoVUL-servoVLL)/2.0f;
  float zMP = zHA + servoVLL;
  timeCnt--;
  if (timeCnt < 1) {
    timeCnt = period;
    if (phase < 1) {phase = 1;} else {phase = 0; cycles++;}
  }
  if (phase < 1) {
    // first half-cycle
    zF = zMP + (zHA * sin(((period - timeCnt)*3.14f)/period));
  } else {
    zF = zMP - (zHA * sin(((timeCnt)*3.14f)/period));
  } servoVal = PApplet.parseInt(zF);
  setXSlider(); sendAngle(); drawFlag = 1;
}

// ----------------------------------------------------------------------

public void doSquare() {
  // send squarewave angles
  timeCnt--;
  if (timeCnt < 1) {
    timeCnt = period;
    if (phase < 1) {
      phase = 1; servoVal = servoVUL;
    } else {
      phase = 0; servoVal = servoVLL; cycles++;
    }
    setXSlider(); sendAngle(); drawFlag = 1;
  }
}

// ----------------------------------------------------------------------

public void doSwing() {
  // send cranked swing wave angles
  timeCnt--;
  if (timeCnt == period/2) {
    servoVal = servoVLL + ((servoVUL - servoVLL)/2);
    setXSlider(); sendAngle(); drawFlag = 1;
  }
  if (timeCnt < 1) {
    timeCnt = period;
    if (phase < 1) {
      phase = 1;
      servoVal = servoVUL;
    } else {
      phase = 0;
      servoVal = servoVLL;
      cycles++;
    }
    setXSlider(); sendAngle(); drawFlag = 1;
  }
}

// ----------------------------------------------------------------------

public void doTriangle() {
  // send triangle waveform angles
  timeCnt--;
  if (timeCnt < 1) {
    timeCnt = period;
    if (phase < 1) {phase = 1;} else {phase = 0; cycles++;}
  }
  if (phase < 1) {
    // first half-cycle
    servoVal = (((servoVUL-servoVLL)*(period-timeCnt))/period) + servoVLL;
  } else {
    servoVal = (((servoVUL-servoVLL)*timeCnt)/period) + servoVLL;
  }
  setXSlider(); sendAngle(); drawFlag = 1;
}

// ----------------------------------------------------------------------

public void drawChMsg() {
  // put text in Ch field
  if (servoPin < 0) {TR = 255;}
  drawTxtField("    ", 462, 34);
  drawTxtField(msgCh, 462, 34); TR = 0;
}

// ----------------------------------------------------------------------

public void drawCOMMsg() {
  // put text in COM value field
  if (comFlag < 1) {TR = 255;}
  drawTxtField(msgCOM, 95, 237); TR = 0;
}

// ----------------------------------------------------------------------

public void drawCycles() {
  // put text in Ch field
  if (ctrlMode < 1) {TR = 255;}
  drawTxtField("              ", 676, 339);
  drawTxtField(str(cycles), 676, 339); TR = 0;
}

// ----------------------------------------------------------------------

public void drawFreq() {
  // put text in Freq. value field
  if (ctrlMode < 1) {TR = 255;}
  drawTxtField(str(period), 695, 305); TR = 0;
}

// ----------------------------------------------------------------------

public void drawJoyX() {
  // draw the X joystick slider
  int zXw = 20; stroke(0,0,0); 
  strokeWeight(2); fill(255,255,255); 
  rect(JoyXval-zXw,159,zXw+zXw,40);
}

// ----------------------------------------------------------------------

public void drawLamps() {
  // draw lamps based on flags
  fill(20,210,0); stroke(0,0,0); strokeWeight(1); // green
  if (angleMode < 1) {
    rect(649,69,9,29);
  } else {
    rect(649,109,9,29);
  }
  if (pingON > 0) {rect(745,109,9,29);}
  fill(255,50,0); // red
  switch(ctrlMode) {
    case 0:
      ellipse(116,296,21,21); 
      if ((quiverFlag > 0) && (qPhase > 0)) {fill(0,100,255); ellipse(216,296,21,21);}
      break;
    case 1:
      ellipse(317,296,21,21); break;
    case 2:
      ellipse(418,296,21,21); break;
    case 3:
      ellipse(518,296,21,21); break;
    case 4:
      ellipse(619,296,21,21); break;
  }
}

// ----------------------------------------------------------------------

public void drawLLMsg() {
  // put text in LL value field
  if (angleMode < 1) {TR = 255;}
  drawTxtField("    ", 66, 172);
  if (angleMode > 0) {
    drawTxtField(str(servoVLL), 66, 172); TR = 0;
  } else {
    drawTxtField(str(PApplet.parseInt((servoVLL-servoVLL)*180)/(servoVUL-servoVLL)), 66, 172); TR = 0;
  } TR = 0;
}

// ----------------------------------------------------------------------

public void drawMouseXY() {
  // draws the value of mouse x,y top left
  mX = mouseX; mY = mouseY;
  fill(200,200,200); rect(10,8,50,30);
  fill(0,0,0); text("X=" + mX, 12, 20); text("Y=" + mY, 12, 34);
}

// ----------------------------------------------------------------------

public void drawRxMsg() {
  // put text in the Rx field
  drawTxtField("                                                       ", 246, 237);
  drawTxtField(msgRx, 246, 237);
}

// ----------------------------------------------------------------------

public void drawServoVal() {
  // put text in Val field
  drawTxtField("         ", 561, 34);
  // normal single channel mode
  if (angleMode > 0) {
    drawTxtField(str(servoVal), 561, 34);
  } else {
    drawTxtField(str(PApplet.parseInt((servoVal-servoVLL)*180)/(servoVUL-servoVLL)), 561, 34);
  }
}

// ----------------------------------------------------------------------

public void drawTxMsg() {
  // put text in Tx field
  drawTxtField("                                                      ", 540, 237);
  drawTxtField(msgTx, 540, 237);
}

// ----------------------------------------------------------------------

public void drawTxtField(String zM,int zX, int zY) {
  stroke(242,242,242); fill(242,242,242); 
  rect(zX, zY, 8+textWidth(zM), 14);
  fill(TR,TG,TB); text(zM, zX+4, zY+12);
}

// ----------------------------------------------------------------------

public void drawULMsg() {
  // put text in UL value field
  if (angleMode < 1) {TR = 255;}
  drawTxtField("    ", 701, 172);
  if (angleMode > 0) {
    drawTxtField(str(servoVUL), 701, 172); TR = 0;
  } else {
    drawTxtField(str(PApplet.parseInt((servoVUL-servoVLL)*180)/(servoVUL-servoVLL)), 701, 172); TR = 0;
  } TR = 0;
}

// ----------------------------------------------------------------------

public void exit() {
  // program is closing so reset Arduino...
  usbPortWrite("!");
  delay(20); // allow 20ms for data to be sent
  super.exit();
}

// ----------------------------------------------------------------------

public void getCOMPort() {
  // initialise serial comms if a port exists
  if (comFlag > 0) {
    // stops an existing connection if already made
    usbPort.stop();
    comFlag = 0; msgCOM = "----"; delay(100);
  }
  comListLength = Serial.list().length;
  if (comListLength > 0) {
    comPnt++; if (comPnt >= comListLength) {comPnt = 0;}
    // test available serial port
    comName = Serial.list()[comPnt];
    println(comName);
    try {
      usbPort = new Serial(this, comName, 19200);
      usbPort.bufferUntil(LF);
      comFlag = 1; msgCOM = comName;
    } catch(Exception e) {
      comFlag = 0; msgCOM = "-Error-";
    }
  } else {
    // no serial port available
    comFlag = 0; msgCOM = "-NA-";
  } refreshFlag = 1;
}

// ----------------------------------------------------------------------

public void incPeriod() {
  // increment the period value
  period++; drawFlag = 1;
}

// ----------------------------------------------------------------------

public void incXSlider() {
  // increment slider value
  if (servoVal < servoVUL) {
    servoLast = servoVal;
    if (angleMode > 0) {
      servoVal++;
    } else {
      zF = servoVal;
      zF = 0.27f + (((zF - 1000.0f) * 180.0f) / 1000.0f); zF++;
      servoVal = PApplet.parseInt((zF * 1000.0f)/180.0f) + 1000; 
      servoVal = min(servoVal,servoVUL);
    } setXSlider(); sendAngle();
  }
}

// ----------------------------------------------------------------------

public void loadScrn(int zMode) {
  // load a background image
  if (zMode == 0) {img = loadImage("Control Pad.png");}
  drawFlag = 1;
}

// ----------------------------------------------------------------------

public void readXSlider() {
  // determine X-slider position and send value
  JoyXval = mX; JoyXval = max(JoyXval, JoyX0);
  JoyXval = min(JoyXval, JoyX1);
  // normal single channel mode
  servoVal = servoVLL + (((JoyXval - JoyX0) * (servoVUL - servoVLL)) / (JoyX1 - JoyX0));
  servoVal = max(servoVal, servoVLL);
  servoVal = min(servoVal, servoVUL);
  if (servoLast != servoVal) {
    servoLast = servoVal;
    sendAngle();
  } servoLast = servoVal;
}

// ----------------------------------------------------------------------

public void resetFlags() {
  // set flags to match RESET condition
  angleMode = 1; // 0 = degrees, 1 = microseconds
  ctrlMode = 0; // control mode, 0 = slider, 1 = square, 2 = triangle, 3 =sine
  cycles = 0; // waveform counter
  msgCh = "---"; // channel message sent to Arduino
  mTO = 0; // mouse down timeout
  period = 60; // half period in draw loop units
  phase = 0; // current waveform phase
  qPhase = 0; // quiver phase, 0 or 1
  quiverFlag = 0; // Quiver mode 0 = OFF, 1 = ON
  qVal = 100; // applied quiver offset
  servoMin = 400; // limit constants for servo values
  servoMax = 2600; // limit constants for servo values
  servoMMD = servoMax - servoMin; // limit difference constant for servo values
  servoPin = -1; // pin number used by Arduino
  servoSwp = 0; // servo sweep flag; default = 0  OFF, 1 = ON
  servoVal = 1472; // =90' servo middle value
  servoVLL = 400; // =0' servo LL value
  servoVUL = 2600; // =180' servo UL value
  sliderDwn = 0; // =1 when mouse is clicked on slider
  timeCnt = 0; // wave count down timer
  TMC = 0; // main loop skip timer
  //
  centreJoyX();
}

// ----------------------------------------------------------------------

public void sendAngle() {
  // send an angle value to the USB port depending on mode
  if (angleMode > 0) {
    msgTx = "SM" + str(servoVal) + ".";
  } else {
    msgTx = "SA" + str(PApplet.parseInt((servoVal-servoVLL)*180)/(servoVUL-servoVLL)) + ".";
  }
  usbPortWrite(msgTx); drawFlag = 1;
}

// ----------------------------------------------------------------------

public void sendButton(String zS) {
  // called when a button is clicked on the scrn
  msgTx = zS; usbPortWrite(msgTx);
  drawFlag = 1;
}

// ----------------------------------------------------------------------

public void sendLL() {
  // send a lower limit msg to arduino
  msgTx = "SL" + str(servoVLL) + "."; usbPortWrite(msgTx);
  drawFlag = 1;
}

// ----------------------------------------------------------------------

public void sendPin() {
  // toggle selected Pin number
  int zPin = (mX-199)/50;
  if (mY > 104) {zPin = zPin + 7;}
  if (servoPin != zPin) {
    // new pin number selected
    sendLL(); sendUL();
    msgCh = str(zPin);
    sendButton("SP" + str(zPin) + ".");
    servoPin = zPin;  
  } else {
    // same pin so toggle OFF
    msgCh = "---";
    sendButton("SD.");
    servoPin = -1;  
  }
  drawFlag = 1;
}

// ----------------------------------------------------------------------

public void sendRESET() {
  // reset defaults and send RESET msg to arduino
  msgTx = "!"; usbPortWrite(msgTx);
  resetFlags(); drawFlag = 1;
}

// ----------------------------------------------------------------------

public void sendUL() {
  // send an upper limit msg to arduino
  msgTx = "SU" + str(servoVUL) + "."; usbPortWrite(msgTx);
  drawFlag = 1;
}

// ----------------------------------------------------------------------

public void setAngleMode(int zA) {
  // called to switch between angle modes
  angleMode = zA; drawFlag = 1;
}

// ----------------------------------------------------------------------

public void setLL() {
  // set lower limit from slider position
  servoVLL = servoVal;
  setXSlider(); sendLL(); calcQV();
}

// ----------------------------------------------------------------------

public void setLLValDwn() {
  // reduce the slider lower limit
  if (angleMode > 0) {
    servoVLL--;
    servoVLL = max(servoMin,servoVLL);
    setXSlider(); sendLL(); calcQV();
  }
}

// ----------------------------------------------------------------------

public void setLLValUp() {
  // increase the slider upper limit
  if (angleMode > 0) {
    servoVLL++;
    if (servoVLL > servoVUL) {servoVUL = servoVLL;}
    servoVLL = min(servoMax,servoVLL);
    servoVUL  = min(servoMax,servoVUL);
    setXSlider(); sendLL(); calcQV();
  }
}

// ----------------------------------------------------------------------

public void setMode(int zM) {
  // set the mode to zM
  phase = 0; period = 60;
  switch(zM) {
    case 0:
      // slider mode
      break;
    case 1:
      // squarewave mode
      break;
    case 2:
      // triangle wave mode
      break;
  } ctrlMode = zM; drawFlag = 1;
}

// ----------------------------------------------------------------------

public void setQuiver() {
  // toggle the quiver if in mode 0
  if (ctrlMode == 0) {
    if (quiverFlag < 1) {
      // turn quiver ON
      quiverFlag = 1; qPhase = 0; calcQV();
    } else {
      // turn quiver OFF and centre value
      quiverFlag = -1;
      msgTx = "SM" + str(servoVal) + ".";
    }
  } drawFlag = 1;
}

// ----------------------------------------------------------------------

public void setQuiverOFF() {
  // turn the quiver mode OFF
  quiverFlag = 0;
  msgTx = "SM" + str(servoVal) + ".";
  usbPortWrite(msgTx); drawFlag = 1;
}

// ----------------------------------------------------------------------

public void setUL() {
  // set upper limit from slider position
  servoVUL = servoVal;
  setXSlider(); sendUL(); calcQV();
}

// ----------------------------------------------------------------------

public void setULValDwn() {
  // reduce the slider upper limit
  if (angleMode > 0) {
    servoVUL--;
    if (servoVUL < servoVLL) {servoVLL = servoVUL;}
    servoVLL = max(servoVLL, servoMin);
    servoVUL = max(servoVUL, servoMin);
    setXSlider(); sendUL(); calcQV();
  }
}

// ----------------------------------------------------------------------

public void setULValUp() {
  // increase the slider upper limit
  if (angleMode > 0) {
      servoVUL++;
      servoVUL = min(servoMax,servoVUL);
      setXSlider(); sendUL(); calcQV();
  }
}

// ----------------------------------------------------------------------

public void setXSlider() {
  // set the X-slider position based on servoVal
  int zVal = servoVal;
  servoVal = min(servoVal, servoVUL);
  servoVal = max(servoVal, servoVLL);
  if (servoVUL > servoVLL) {
    JoyXval = 170 + ((servoVal - servoVLL) * (630 - 170))/(servoVUL - servoVLL);
  }
  JoyXval = max(170, JoyXval); // prevent it going too low
  JoyXval = min(630, JoyXval); // prevent it going too high
  if (zVal != servoVal) {
    servoLast = servoVal;
    sendAngle();
  }
  drawFlag = 1;
}

//-------------------------------------------------------------------------

public void toggleMouseON() {
  // toggle mouse ON/OFF flag
  if (mON > 0) {
    mON = 0;
  } else {
    mON = 1;
  } drawFlag = 1;
}

// ----------------------------------------------------------------------

public void togglePing() {
  // toggle ping flag ON/OFF
  if (pingON > 0) {
    pingON = 0;
  } else {
    pingON = 1;
  }
  drawFlag = 1;
}

// ----------------------------------------------------------------------

public void usbPortWrite(String zmsg) {
  // called to send text over the serial port
  if (comFlag > 0) {usbPort.write(zmsg);}
}
// ----------------------------------------------------------------------
  public void settings() {  size(800,408); }
  static public void main(String[] passedArgs) {
    String[] appletArgs = new String[] { "Proc_Servo_Ctrl_05" };
    if (passedArgs != null) {
      PApplet.main(concat(appletArgs, passedArgs));
    } else {
      PApplet.main(appletArgs);
    }
  }
}
